/**
 * YouTube Fancy Text Extension - Production Version
 * Enhanced with error handling, browser compatibility, and performance optimizations
 */

(function() {
  'use strict';

  // Global variables
  let isInitialized = false;
  let extensionPanel = null;
  let toggleButton = null;
  let eventListeners = [];

  // Utility functions for browser compatibility
  function safeAddEventListener(element, event, handler) {
    element.addEventListener(event, handler);
    eventListeners.push({element: element, event: event, handler: handler});
  }

  function cleanupEventListeners() {
    eventListeners.forEach(function(listener) {
      listener.element.removeEventListener(listener.event, listener.handler);
    });
    eventListeners = [];
  }

  // Input validation and sanitization
  function validateInput(text) {
    if (typeof text !== 'string') return '';
    if (text.length > 500) return text.substring(0, 500);
    return text.trim();
  }

  // Safe clipboard fallback for older browsers
  function copyToClipboard(text) {
    // Modern browsers
    if (navigator.clipboard && navigator.clipboard.writeText) {
      return navigator.clipboard.writeText(text).then(function() {
        return true;
      }).catch(function() {
        return fallbackCopyToClipboard(text);
      });
    } else {
      return Promise.resolve(fallbackCopyToClipboard(text));
    }
  }

  function fallbackCopyToClipboard(text) {
    var textArea = document.createElement('textarea');
    textArea.value = text;
    textArea.style.position = 'fixed';
    textArea.style.left = '-999999px';
    textArea.style.top = '-999999px';
    document.body.appendChild(textArea);
    textArea.focus();
    textArea.select();
    
    try {
      var result = document.execCommand('copy');
      document.body.removeChild(textArea);
      return result;
    } catch (err) {
      document.body.removeChild(textArea);
      return false;
    }
  }

  // Optimized text transformation functions
  var fancyTextStyles = {
    bold: {
      name: "𝐁𝐨𝐥𝐝",
      transform: function(text) {
        var normalChars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";
        var boldChars = "𝐀𝐁𝐂𝐃𝐄𝐅𝐆𝐇𝐈𝐉𝐊𝐋𝐌𝐍𝐎𝐏𝐐𝐑𝐒𝐓𝐔𝐕𝐖𝐗𝐘𝐙𝐚𝐛𝐜𝐝𝐞𝐟𝐠𝐡𝐢𝐣𝐤𝐥𝐦𝐧𝐨𝐩𝐪𝐫𝐬𝐭𝐮𝐯𝐰𝐱𝐲𝐳𝟎𝟏𝟐𝟑𝟒𝟓𝟔𝟕𝟖𝟗";
        return transformText(text, normalChars, boldChars);
      }
    },
    italic: {
      name: "𝐼𝑡𝑎𝑙𝑖𝑐",
      transform: function(text) {
        var normalChars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
        var italicChars = "𝐴𝐵𝐶𝐷𝐸𝐹𝐺𝐻𝐼𝐽𝐾𝐿𝑀𝑁𝑂𝑃𝑄𝑅𝑆𝑇𝑈𝑉𝑊𝑋𝑌𝑍𝑎𝑏𝑐𝑑𝑒𝑓𝑔ℎ𝑖𝑗𝑘𝑙𝑚𝑛𝑜𝑝𝑞𝑟𝑠𝑡𝑢𝑣𝑤𝑥𝑦𝑧";
        return transformText(text, normalChars, italicChars);
      }
    },
    script: {
      name: "𝒮𝒸𝓇𝒾𝓅𝓉",
      transform: function(text) {
        var normalChars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
        var scriptChars = "𝒜ℬ𝒞𝒟ℰℱ𝒢ℋℐ𝒥𝒦ℒℳ𝒩𝒪𝒫𝒬ℛ𝒮𝒯𝒰𝒱𝒲𝒳𝒴𝒵𝒶𝒷𝒸𝒹ℯ𝒻ℊ𝒽𝒾𝒿𝓀𝓁𝓂𝓃ℴ𝓅𝓆𝓇𝓈𝓉𝓊𝓋𝓌𝓍𝓎𝓏";
        return transformText(text, normalChars, scriptChars);
      }
    },
    monospace: {
      name: "𝙼𝚘𝚗𝚘𝚜𝚙𝚊𝚌𝚎",
      transform: function(text) {
        var normalChars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";
        var monospaceChars = "𝙰𝙱𝙲𝙳𝙴𝙵𝙶𝙷𝙸𝙹𝙺𝙻𝙼𝙽𝙾𝙿𝚀𝚁𝚂𝚃𝚄𝚅𝚆𝚇𝚈𝚉𝚊𝚋𝚌𝚍𝚎𝚏𝚐𝚑𝚒𝚓𝚔𝚕𝚖𝚗𝚘𝚙𝚚𝚛𝚜𝚝𝚞𝚟𝚠𝚡𝚢𝚣𝟶𝟷𝟸𝟹𝟺𝟻𝟼𝟽𝟾𝟿";
        return transformText(text, normalChars, monospaceChars);
      }
    },
    smallcaps: {
      name: "Sᴍᴀʟʟ Cᴀᴘs",
      transform: function(text) {
        var normalChars = "abcdefghijklmnopqrstuvwxyz";
        var smallCapsChars = "ᴀʙᴄᴅᴇғɢʜɪᴊᴋʟᴍɴᴏᴘǫʀsᴛᴜᴠᴡxʏᴢ";
        return transformText(text, normalChars, smallCapsChars);
      }
    },
    bubble: {
      name: "Ⓑⓤⓑⓑⓛⓔ",
      transform: function(text) {
        var bubbleMap = {
          'a': 'ⓐ', 'b': 'ⓑ', 'c': 'ⓒ', 'd': 'ⓓ', 'e': 'ⓔ', 'f': 'ⓕ', 'g': 'ⓖ', 'h': 'ⓗ',
          'i': 'ⓘ', 'j': 'ⓙ', 'k': 'ⓚ', 'l': 'ⓛ', 'm': 'ⓜ', 'n': 'ⓝ', 'o': 'ⓞ', 'p': 'ⓟ',
          'q': 'ⓠ', 'r': 'ⓡ', 's': 'ⓢ', 't': 'ⓣ', 'u': 'ⓤ', 'v': 'ⓥ', 'w': 'ⓦ', 'x': 'ⓧ',
          'y': 'ⓨ', 'z': 'ⓩ', 'A': 'Ⓐ', 'B': 'Ⓑ', 'C': 'Ⓒ', 'D': 'Ⓓ', 'E': 'Ⓔ', 'F': 'Ⓕ',
          'G': 'Ⓖ', 'H': 'Ⓗ', 'I': 'Ⓘ', 'J': 'Ⓙ', 'K': 'Ⓚ', 'L': 'Ⓛ', 'M': 'Ⓜ', 'N': 'Ⓝ',
          'O': 'Ⓞ', 'P': 'Ⓟ', 'Q': 'Ⓠ', 'R': 'Ⓡ', 'S': 'Ⓢ', 'T': 'Ⓣ', 'U': 'Ⓤ', 'V': 'Ⓥ',
          'W': 'Ⓦ', 'X': 'Ⓧ', 'Y': 'Ⓨ', 'Z': 'Ⓩ'
        };
        return transformWithMap(text, bubbleMap);
      }
    },
    negativebubble: {
      name: "🅝🅔🅖🅐🅣🅘🅥🅔 🅑🅤🅑🅑🅛🅔",
      transform: function(text) {
        var negBubbleMap = {
          'a': '🅐', 'b': '🅑', 'c': '🅒', 'd': '🅓', 'e': '🅔', 'f': '🅕', 'g': '🅖', 'h': '🅗',
          'i': '🅘', 'j': '🅙', 'k': '🅚', 'l': '🅛', 'm': '🅜', 'n': '🅝', 'o': '🅞', 'p': '🅟',
          'q': '🅠', 'r': '🅡', 's': '🅢', 't': '🅣', 'u': '🅤', 'v': '🅥', 'w': '🅦', 'x': '🅧',
          'y': '🅨', 'z': '🅩', 'A': '🅐', 'B': '🅑', 'C': '🅒', 'D': '🅓', 'E': '🅔', 'F': '🅕',
          'G': '🅖', 'H': '🅗', 'I': '🅘', 'J': '🅙', 'K': '🅚', 'L': '🅛', 'M': '🅜', 'N': '🅝',
          'O': '🅞', 'P': '🅟', 'Q': '🅠', 'R': '🅡', 'S': '🅢', 'T': '🅣', 'U': '🅤', 'V': '🅥',
          'W': '🅦', 'X': '🅧', 'Y': '🅨', 'Z': '🅩'
        };
        return transformWithMap(text, negBubbleMap);
      }
    },
    circled: {
      name: "Ⓒⓘⓡⓒⓛⓔⓓ",
      transform: function(text) {
        var normalChars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";
        var circledChars = "ⒶⒷⒸⒹⒺⒻⒼⒽⒾⒿⓀⓁⓂⓃⓄⓅⓆⓇⓈⓉⓊⓋⓌⓍⓎⓏⓐⓑⓒⓓⓔⓕⓖⓗⓘⓙⓚⓛⓜⓝⓞⓟⓠⓡⓢⓣⓤⓥⓦⓧⓨⓩ⓪①②③④⑤⑥⑦⑧⑨";
        return transformText(text, normalChars, circledChars);
      }
    },
    squared: {
      name: "🆂🆀🆄🅰🆁🅴🅳",
      transform: function(text) {
        var normalChars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        var squaredChars = "🅰🅱🅲🅳🅴🅵🅶🅷🅸🅹🅺🅻🅼🅽🅾🅿🆀🆁🆂🆃🆄🆅🆆🆇🆈🆉";
        return transformText(text.toUpperCase(), normalChars, squaredChars);
      }
    },
    parenthesized: {
      name: "⒫⒜⒭⒠⒩⒯⒣⒠⒮⒤⒵⒠⒟",
      transform: function(text) {
        var parenMap = {
          'a': '⒜', 'b': '⒝', 'c': '⒞', 'd': '⒟', 'e': '⒠', 'f': '⒡', 'g': '⒢', 'h': '⒣',
          'i': '⒤', 'j': '⒥', 'k': '⒦', 'l': '⒧', 'm': '⒨', 'n': '⒩', 'o': '⒪', 'p': '⒫',
          'q': '⒬', 'r': '⒭', 's': '⒮', 't': '⒯', 'u': '⒰', 'v': '⒱', 'w': '⒲', 'x': '⒳',
          'y': '⒴', 'z': '⒵', 'A': '⒜', 'B': '⒝', 'C': '⒞', 'D': '⒟', 'E': '⒠', 'F': '⒡',
          'G': '⒢', 'H': '⒣', 'I': '⒤', 'J': '⒥', 'K': '⒦', 'L': '⒧', 'M': '⒨', 'N': '⒩',
          'O': '⒪', 'P': '⒫', 'Q': '⒬', 'R': '⒭', 'S': '⒮', 'T': '⒯', 'U': '⒰', 'V': '⒱',
          'W': '⒲', 'X': '⒳', 'Y': '⒴', 'Z': '⒵'
        };
        return transformWithMap(text, parenMap);
      }
    },
    currency: {
      name: "₵ɄɌɌɆ₦₵Ɏ",
      transform: function(text) {
        var currencyMap = {
          'a': '₳', 'b': '฿', 'c': '₵', 'd': 'Đ', 'e': '€', 'f': '₣', 'g': 'Ꮆ', 'h': 'Ⱨ',
          'i': 'ł', 'j': 'J', 'k': '₭', 'l': 'Ⱡ', 'm': '₥', 'n': '₦', 'o': 'Ø', 'p': '₱',
          'q': 'Q', 'r': 'Ɽ', 's': '₴', 't': '₮', 'u': 'Ʉ', 'v': 'V', 'w': '₩', 'x': 'Ӿ',
          'y': '¥', 'z': 'Ⱬ', 'A': '₳', 'B': '฿', 'C': '₵', 'D': 'Đ', 'E': '€', 'F': '₣',
          'G': 'Ꮆ', 'H': 'Ⱨ', 'I': 'ł', 'J': 'J', 'K': '₭', 'L': 'Ⱡ', 'M': '₥', 'N': '₦',
          'O': 'Ø', 'P': '₱', 'Q': 'Q', 'R': 'Ɽ', 'S': '₴', 'T': '₮', 'U': 'Ʉ', 'V': 'V',
          'W': '₩', 'X': 'Ӿ', 'Y': '¥', 'Z': 'Ⱬ'
        };
        return transformWithMap(text, currencyMap);
      }
    },
    greek: {
      name: "Gяεεк",
      transform: function(text) {
        var greekMap = {
          'a': 'α', 'b': 'β', 'c': 'ς', 'd': 'δ', 'e': 'ε', 'f': 'φ', 'g': 'γ', 'h': 'η',
          'i': 'ι', 'j': 'ј', 'k': 'κ', 'l': 'λ', 'm': 'μ', 'n': 'ν', 'o': 'σ', 'p': 'ρ',
          'q': 'φ', 'r': 'ρ', 's': 'ѕ', 't': 'τ', 'u': 'υ', 'v': 'ν', 'w': 'ω', 'x': 'χ',
          'y': 'ψ', 'z': 'ζ', 'A': 'Α', 'B': 'Β', 'C': 'Ϲ', 'D': 'Δ', 'E': 'Ε', 'F': 'Φ',
          'G': 'Γ', 'H': 'Η', 'I': 'Ι', 'J': 'Ј', 'K': 'Κ', 'L': 'Λ', 'M': 'Μ', 'N': 'Ν',
          'O': 'Ο', 'P': 'Ρ', 'Q': 'Q', 'R': 'R', 'S': 'Ѕ', 'T': 'Τ', 'U': 'Υ', 'V': 'V',
          'W': 'Ω', 'X': 'Χ', 'Y': 'Ψ', 'Z': 'Ζ'
        };
        return transformWithMap(text, greekMap);
      }
    },
    inverted: {
      name: "pǝʇɹǝʌuI",
      transform: function(text) {
        var invertedMap = {
          'a': 'ɐ', 'b': 'q', 'c': 'ɔ', 'd': 'p', 'e': 'ǝ', 'f': 'ɟ', 'g': 'ƃ', 'h': 'ɥ',
          'i': 'ᴉ', 'j': 'ɾ', 'k': 'ʞ', 'l': 'l', 'm': 'ɯ', 'n': 'u', 'o': 'o', 'p': 'd',
          'q': 'b', 'r': 'ɹ', 's': 's', 't': 'ʇ', 'u': 'n', 'v': 'ʌ', 'w': 'ʍ', 'x': 'x',
          'y': 'ʎ', 'z': 'z', 'A': '∀', 'B': 'ᗺ', 'C': 'Ↄ', 'D': 'ᗡ', 'E': 'Ǝ', 'F': 'ᖴ',
          'G': 'פ', 'H': 'H', 'I': 'I', 'J': 'ſ', 'K': 'ʞ', 'L': '˥', 'M': 'W', 'N': 'N',
          'O': 'O', 'P': 'Ԁ', 'Q': 'Q', 'R': 'ᴿ', 'S': 'S', 'T': '┴', 'U': '∩', 'V': 'Λ',
          'W': 'M', 'X': 'X', 'Y': '⅄', 'Z': 'Z'
        };
        var result = '';
        for (var i = 0; i < text.length; i++) {
          result = (invertedMap[text[i]] || text[i]) + result;
        }
        return result;
      }
    },
    superscript: {
      name: "ˢᵘᵖᵉʳˢᶜʳⁱᵖᵗ",
      transform: function(text) {
        var superMap = {
          'a': 'ᵃ', 'b': 'ᵇ', 'c': 'ᶜ', 'd': 'ᵈ', 'e': 'ᵉ', 'f': 'ᶠ', 'g': 'ᵍ', 'h': 'ʰ',
          'i': 'ⁱ', 'j': 'ʲ', 'k': 'ᵏ', 'l': 'ˡ', 'm': 'ᵐ', 'n': 'ⁿ', 'o': 'ᵒ', 'p': 'ᵖ',
          'q': 'q', 'r': 'ʳ', 's': 'ˢ', 't': 'ᵗ', 'u': 'ᵘ', 'v': 'ᵛ', 'w': 'ʷ', 'x': 'ˣ',
          'y': 'ʸ', 'z': 'ᶻ', 'A': 'ᴬ', 'B': 'ᴮ', 'C': 'ᶜ', 'D': 'ᴰ', 'E': 'ᴱ', 'F': 'ᶠ',
          'G': 'ᴳ', 'H': 'ᴴ', 'I': 'ᴵ', 'J': 'ᴶ', 'K': 'ᴷ', 'L': 'ᴸ', 'M': 'ᴹ', 'N': 'ᴺ',
          'O': 'ᴼ', 'P': 'ᴾ', 'Q': 'Q', 'R': 'ᴿ', 'S': 'ˢ', 'T': 'ᵀ', 'U': 'ᵁ', 'V': 'ⱽ',
          'W': 'ᵂ', 'X': 'ˣ', 'Y': 'ʸ', 'Z': 'ᶻ', '0': '⁰', '1': '¹', '2': '²', '3': '³',
          '4': '⁴', '5': '⁵', '6': '⁶', '7': '⁷', '8': '⁸', '9': '⁹'
        };
        return transformWithMap(text, superMap);
      }
    },
    subscript: {
      name: "ₛᵤᵦₛcᵣᵢₚₜ",
      transform: function(text) {
        var subMap = {
          'a': 'ₐ', 'e': 'ₑ', 'h': 'ₕ', 'i': 'ᵢ', 'j': 'ⱼ', 'k': 'ₖ', 'l': 'ₗ', 'm': 'ₘ', 
          'n': 'ₙ', 'o': 'ₒ', 'p': 'ₚ', 'r': 'ᵣ', 's': 'ₛ', 't': 'ₜ', 'u': 'ᵤ', 'v': 'ᵥ', 'x': 'ₓ',
          '0': '₀', '1': '₁', '2': '₂', '3': '₃', '4': '₄', '5': '₅',
          '6': '₆', '7': '₇', '8': '₈', '9': '₉'
        };
        return transformWithMap(text, subMap);
      }
    },
    wide: {
      name: "Ｗｉｄｅ",
      transform: function(text) {
        var result = '';
        for (var i = 0; i < text.length; i++) {
          var code = text.charCodeAt(i);
          if (code >= 33 && code <= 126) {
            result += String.fromCharCode(code - 33 + 0xFF01);
          } else {
            result += text[i];
          }
        }
        return result;
      }
    },
    strikethrough: {
      name: "S̶t̶r̶i̶k̶e̶t̶h̶r̶o̶u̶g̶h̶",
      transform: function(text) {
        var result = '';
        for (var i = 0; i < text.length; i++) {
          result += text[i] + '\u0336';
        }
        return result;
      }
    },
    underline: {
      name: "U̲n̲d̲e̲r̲l̲i̲n̲e̲",
      transform: function(text) {
        var result = '';
        for (var i = 0; i < text.length; i++) {
          result += text[i] + '\u0332';
        }
        return result;
      }
    },
    zalgomini: {
      name: "Z̈ả̧l̤g͈o̩ ̧M̪i̝n̗i̱",
      transform: function(text) {
        var zalgoUp = ['\u030d', '\u030e', '\u0304', '\u0305', '\u033f', '\u0311', '\u0306', '\u0310'];
        var zalgoDown = ['\u0316', '\u0317', '\u0318', '\u0319', '\u031c', '\u031d', '\u031e', '\u031f'];
        var result = '';
        for (var i = 0; i < text.length; i++) {
          result += text[i];
          if (Math.random() < 0.3) result += zalgoUp[Math.floor(Math.random() * zalgoUp.length)];
          if (Math.random() < 0.3) result += zalgoDown[Math.floor(Math.random() * zalgoDown.length)];
        }
        return result;
      }
    },
    zalgo: {
      name: "Z̸̰̈́å̸̰l̷̰̈́g̸̰̈́ö̶̰́",
      transform: function(text) {
        var zalgoUp = ['\u030d', '\u030e', '\u0304', '\u0305', '\u033f', '\u0311', '\u0306', '\u0310'];
        var zalgoMid = ['\u0315', '\u031b', '\u0340', '\u0341', '\u0358', '\u0321', '\u0322', '\u0327'];
        var zalgoDown = ['\u0316', '\u0317', '\u0318', '\u0319', '\u031c', '\u031d', '\u031e', '\u031f'];
        var result = '';
        for (var i = 0; i < text.length; i++) {
          result += text[i];
          for (var j = 0; j < Math.floor(Math.random() * 3); j++) {
            result += zalgoUp[Math.floor(Math.random() * zalgoUp.length)];
          }
          if (Math.random() < 0.3) result += zalgoMid[Math.floor(Math.random() * zalgoMid.length)];
          for (var j = 0; j < Math.floor(Math.random() * 3); j++) {
            result += zalgoDown[Math.floor(Math.random() * zalgoDown.length)];
          }
        }
        return result;
      }
    },
    leetbasic: {
      name: "L337 84515",
      transform: function(text) {
        var leetMap = {
          'a': '4', 'A': '4', 'e': '3', 'E': '3', 'i': '1', 'I': '1', 'o': '0', 'O': '0',
          's': '5', 'S': '5', 't': '7', 'T': '7', 'l': '1', 'L': '1', 'z': '2', 'Z': '2',
          'g': '9', 'G': '9', 'b': '8', 'B': '8'
        };
        return transformWithMap(text, leetMap);
      }
    },
    leet: {
      name: "|_337 4|)\\|/4|\\/|3|)",
      transform: function(text) {
        var leetMap = {
          'a': '@', 'A': '@', 'b': '|3', 'B': '|3', 'c': '(', 'C': '(', 'd': '|)', 'D': '|)',
          'e': '3', 'E': '3', 'f': '|=', 'F': '|=', 'g': '6', 'G': '6', 'h': '|-|', 'H': '|-|',
          'i': '!', 'I': '!', 'j': '_|', 'J': '_|', 'k': '|<', 'K': '|<', 'l': '|_', 'L': '|_',
          'm': '|\\/|', 'M': '|\\/|', 'n': '|\\|', 'N': '|\\|', 'o': '0', 'O': '0', 'p': '|*', 'P': '|*',
          'q': '0_', 'Q': '0_', 'r': '|2', 'R': '|2', 's': '$', 'S': '$', 't': '+', 'T': '+',
          'u': '|_|', 'U': '|_|', 'v': '\\/', 'V': '\\/', 'w': '\\/\\/', 'W': '\\/\\/', 'x': '><', 'X': '><',
          'y': '`/', 'Y': '`/', 'z': '2', 'Z': '2'
        };
        return transformWithMap(text, leetMap);
      }
    },
    morse: {
      name: "-- --- .-. ... .",
      transform: function(text) {
        var morseMap = {
          'a': '.-', 'b': '-...', 'c': '-.-.', 'd': '-..', 'e': '.', 'f': '..-.', 'g': '--.', 'h': '....',
          'i': '..', 'j': '.---', 'k': '-.-', 'l': '.-..', 'm': '--', 'n': '-.', 'o': '---', 'p': '.--.',
          'q': '--.-', 'r': '.-.', 's': '...', 't': '-', 'u': '..-', 'v': '...-', 'w': '.--', 'x': '-..-',
          'y': '-.--', 'z': '--..', '1': '.----', '2': '..---', '3': '...--', '4': '....-', '5': '.....',
          '6': '-....', '7': '--...', '8': '---..', '9': '----.', '0': '-----', ' ': '/'
        };
        var result = [];
        for (var i = 0; i < text.length; i++) {
          result.push(morseMap[text[i].toLowerCase()] || text[i]);
        }
        return result.join(' ');
      }
    },
    boxed: {
      name: "[B][o][x][e][d]",
      transform: function(text) {
        var result = '';
        for (var i = 0; i < text.length; i++) {
          result += '[' + text[i] + ']';
        }
        return result;
      }
    },
    regional: {
      name: "🇷🇪🇬🇮🇴🇳🇦🇱",
      transform: function(text) {
        var result = '';
        for (var i = 0; i < text.length; i++) {
          var code = text[i].toUpperCase().charCodeAt(0);
          if (code >= 65 && code <= 90) {
            result += String.fromCodePoint(0x1F1E6 + code - 65);
          } else {
            result += text[i];
          }
        }
        return result;
      }
    },
    flipped: {
      name: "pǝddᴉlℲ",
      transform: function(text) {
        var normalChars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789?!.,";
        var flippedChars = "∀qƆpƎℲƃHIſʞ˥WNOԀQɹS┴∩ΛMX⅄Zɐqɔpǝɟƃɥᴉɾʞlɯuodbɹsʇnʌʍxʎz0ƖᄅƐㄣϛ9ㄥ86¿¡'˙";
        var result = '';
        for (var i = text.length - 1; i >= 0; i--) {
          var index = normalChars.indexOf(text[i]);
          result += index !== -1 ? flippedChars[index] : text[i];
        }
        return result;
      }
    }
  };

  // Optimized transformation helper functions
  function transformText(text, normalChars, transformedChars) {
    var result = '';
    for (var i = 0; i < text.length; i++) {
      var index = normalChars.indexOf(text[i]);
      result += index !== -1 ? transformedChars[index] : text[i];
    }
    return result;
  }

  function transformWithMap(text, charMap) {
    var result = '';
    for (var i = 0; i < text.length; i++) {
      result += charMap[text[i]] || text[i];
    }
    return result;
  }

  // Enhanced YouTube chat input detection
  function findYouTubeChatInput() {
    var selectors = [
      // Most current YouTube live chat selectors (2025)
      'yt-live-chat-text-input-field-renderer div[contenteditable="true"]',
      'yt-live-chat-text-input-field-renderer #input',
      'div[id="input"][contenteditable="true"]',
      'div#input.yt-live-chat-text-input-field-renderer',
      '#input.yt-live-chat-text-input-field-renderer',
      // Alternative patterns for different YouTube layouts
      'div[contenteditable="true"][aria-label*="Chat"]',
      'div[contenteditable="true"][aria-label*="Say something"]',
      'div[contenteditable="true"][data-testid*="chat"]',
      'div[contenteditable="true"][placeholder*="Say something"]',
      // Broader YouTube chat patterns
      'yt-live-chat-app div[contenteditable="true"]',
      'yt-live-chat-renderer div[contenteditable="true"]',
      'div[contenteditable="true"][role="textbox"]',
      // Legacy and fallback selectors
      'input[placeholder*="Say something" i]',
      'input[placeholder*="chat" i]',
      'textarea[placeholder*="chat" i]',
      'div[contenteditable="true"][data-a-target="chat-input"]',
      '#chatroom-input',
      // Very broad fallbacks (last resort)
      'div[contenteditable="true"]#input',
      'div[contenteditable="true"]'
    ];

    for (var i = 0; i < selectors.length; i++) {
      try {
        var element = document.querySelector(selectors[i]);
        if (element && (element.offsetWidth > 0 || element.offsetHeight > 0)) {
          console.log('Found chat input with selector:', selectors[i]);
          return element;
        }
      } catch (e) {
        console.warn('Selector failed:', selectors[i], e);
      }
    }
    
    // Additional debugging - log what chat elements we can find
    console.warn('Chat input not found. Available elements:');
    var debugSelectors = [
      'yt-live-chat-text-input-field-renderer',
      'yt-live-chat-app',
      'yt-live-chat-renderer',
      'div[contenteditable]',
      '[aria-label*="chat" i]',
      '[placeholder*="say" i]'
    ];
    
    debugSelectors.forEach(function(selector) {
      var elements = document.querySelectorAll(selector);
      if (elements.length > 0) {
        console.log('Found', elements.length, 'elements for:', selector);
      }
    });
    
    return null;
  }

  // Enhanced text insertion with better error handling
  function insertTextIntoChat(text) {
    try {
      var validatedText = validateInput(text);
      if (!validatedText) {
        showNotification('Please enter some text first!');
        return;
      }

      var chatInput = findYouTubeChatInput();
      
      if (chatInput) {
        // Handle different input types with better compatibility
        if (chatInput.tagName === 'INPUT' || chatInput.tagName === 'TEXTAREA') {
          chatInput.value = validatedText;
          fireInputEvents(chatInput);
        } else if (chatInput.isContentEditable || chatInput.contentEditable === 'true') {
          chatInput.textContent = validatedText;
          fireInputEvents(chatInput);
        }
        
        chatInput.focus();
        showNotification('Text inserted successfully!');
      } else {
        // Try waiting for chat to load, then retry once
        setTimeout(function() {
          var retryInput = findYouTubeChatInput();
          if (retryInput) {
            if (retryInput.tagName === 'INPUT' || retryInput.tagName === 'TEXTAREA') {
              retryInput.value = validatedText;
              fireInputEvents(retryInput);
            } else if (retryInput.isContentEditable || retryInput.contentEditable === 'true') {
              retryInput.textContent = validatedText;
              fireInputEvents(retryInput);
            }
            retryInput.focus();
            showNotification('Text inserted successfully (retry)!');
          } else {
            // Enhanced fallback with better user feedback
            copyToClipboard(validatedText).then(function(success) {
              if (success) {
                showNotification('Chat input not found. Text copied to clipboard - paste it manually! (Try refreshing if on a live stream)');
              } else {
                showNotification('Could not find chat input or copy to clipboard. Text: ' + validatedText);
              }
            });
          }
        }, 1000);
      }
    } catch (error) {
      console.error('Error inserting text:', error);
      showNotification('Error inserting text. Please try again.');
    }
  }

  function fireInputEvents(element) {
    try {
      // Trigger events for better compatibility
      var events = ['input', 'change', 'keydown', 'keyup'];
      events.forEach(function(eventType) {
        var event;
        if (typeof Event === 'function') {
          event = new Event(eventType, { bubbles: true, cancelable: true });
        } else {
          // IE compatibility
          event = document.createEvent('Event');
          event.initEvent(eventType, true, true);
        }
        element.dispatchEvent(event);
      });
    } catch (error) {
      console.warn('Error firing events:', error);
    }
  }

  // Enhanced notification system
  function showNotification(message) {
    try {
      // Remove existing notifications
      var existingNotifications = document.querySelectorAll('.fancy-text-notification');
      existingNotifications.forEach(function(notif) {
        if (notif.parentNode) {
          notif.parentNode.removeChild(notif);
        }
      });

      var notification = document.createElement('div');
      notification.className = 'fancy-text-notification';
      notification.textContent = message;
      notification.style.cssText = 
        'position: fixed; top: 20px; right: 20px; background: #ff6b6b; color: white; ' +
        'padding: 12px 16px; border-radius: 8px; z-index: 10001; font-family: Arial, sans-serif; ' +
        'font-size: 14px; max-width: 300px; box-shadow: 0 4px 12px rgba(0,0,0,0.3); ' +
        'animation: fadeIn 0.3s ease-in;';
      
      document.body.appendChild(notification);
      
      setTimeout(function() {
        if (notification.parentNode) {
          notification.style.animation = 'fadeOut 0.3s ease-out';
          setTimeout(function() {
            if (notification.parentNode) {
              notification.parentNode.removeChild(notification);
            }
          }, 300);
        }
      }, 3000);
    } catch (error) {
      console.error('Error showing notification:', error);
    }
  }

  // Enhanced UI creation with better positioning and mobile support
  function createFancyTextUI() {
    try {
      var panel = document.createElement('div');
      panel.id = 'fancy-text-panel';
      panel.style.cssText = 
        'position: fixed; top: 50%; right: 20px; transform: translateY(-50%); ' +
        'background: rgba(0, 0, 0, 0.95); border: 2px solid #ff6b6b; border-radius: 12px; ' +
        'padding: 16px; z-index: 999999; max-width: 280px; min-width: 250px; ' +
        'box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5); font-family: Arial, sans-serif; ' +
        'display: none; backdrop-filter: blur(10px);';

      var title = document.createElement('h3');
      title.textContent = 'Fancy Text Styles';
      title.style.cssText = 
        'color: #ff6b6b; margin: 0 0 12px 0; font-size: 16px; text-align: center; ' +
        'font-weight: bold;';
      panel.appendChild(title);

      // Enhanced website link
      var websiteLink = document.createElement('div');
      websiteLink.style.cssText = 'text-align: center; margin-bottom: 16px;';
      
      var link = document.createElement('a');
      link.href = 'https://kilaz.com/fancytext/';
      link.target = '_blank';
      link.rel = 'noopener noreferrer';
      link.textContent = '🌐 Visit Full Website - 23+ More Styles!';
      link.style.cssText = 
        'color: #4CAF50; text-decoration: none; font-size: 11px; padding: 6px 10px; ' +
        'border: 1px solid #4CAF50; border-radius: 6px; transition: all 0.3s ease; ' +
        'display: inline-block; background: rgba(76, 175, 80, 0.1);';
      
      safeAddEventListener(link, 'mouseover', function() {
        link.style.background = '#4CAF50';
        link.style.color = '#fff';
      });
      
      safeAddEventListener(link, 'mouseout', function() {
        link.style.background = 'rgba(76, 175, 80, 0.1)';
        link.style.color = '#4CAF50';
      });
      
      websiteLink.appendChild(link);
      panel.appendChild(websiteLink);

      // Enhanced input field
      var inputContainer = document.createElement('div');
      inputContainer.style.marginBottom = '16px';
      
      var input = document.createElement('input');
      input.type = 'text';
      input.placeholder = 'Type your text here...';
      input.id = 'fancy-text-input';
      input.maxLength = 500;
      input.style.cssText = 
        'width: 100%; padding: 10px; border: 1px solid #666; border-radius: 6px; ' +
        'background: #333; color: white; font-size: 14px; box-sizing: border-box; ' +
        'transition: border-color 0.3s ease;';
      
      safeAddEventListener(input, 'focus', function() {
        input.style.borderColor = '#ff6b6b';
      });
      
      safeAddEventListener(input, 'blur', function() {
        input.style.borderColor = '#666';
      });
      
      inputContainer.appendChild(input);
      panel.appendChild(inputContainer);

      // Enhanced styles container with scroll
      var stylesContainer = document.createElement('div');
      stylesContainer.style.cssText = 
        'max-height: 320px; overflow-y: auto; padding-right: 4px;';

      // Create style buttons with improved layout
      var styleKeys = Object.keys(fancyTextStyles);
      for (var i = 0; i < styleKeys.length; i++) {
        var key = styleKeys[i];
        var style = fancyTextStyles[key];
        
        var button = document.createElement('button');
        button.textContent = style.name;
        button.style.cssText = 
          'display: block; width: 100%; margin: 4px 0; padding: 10px 8px; ' +
          'background: #444; color: white; border: 1px solid #666; border-radius: 6px; ' +
          'cursor: pointer; font-size: 12px; transition: all 0.2s ease; ' +
          'text-align: left; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;';
        
        button.setAttribute('data-style', key);
        
        safeAddEventListener(button, 'mouseenter', function() {
          this.style.background = '#555';
          this.style.borderColor = '#ff6b6b';
        });
        
        safeAddEventListener(button, 'mouseleave', function() {
          this.style.background = '#444';
          this.style.borderColor = '#666';
        });

        safeAddEventListener(button, 'click', function() {
          try {
            var inputElement = document.getElementById('fancy-text-input');
            var inputText = validateInput(inputElement.value);
            
            if (inputText) {
              var styleKey = this.getAttribute('data-style');
              var styleObj = fancyTextStyles[styleKey];
              var transformedText = styleObj.transform(inputText);
              insertTextIntoChat(transformedText);
              inputElement.value = '';
            } else {
              showNotification('Please enter some text first!');
              inputElement.focus();
            }
          } catch (error) {
            console.error('Error transforming text:', error);
            showNotification('Error transforming text. Please try again.');
          }
        });

        stylesContainer.appendChild(button);
      }

      panel.appendChild(stylesContainer);

      // Enhanced close button
      var closeButton = document.createElement('button');
      closeButton.innerHTML = '&times;';
      closeButton.title = 'Close panel';
      closeButton.style.cssText = 
        'position: absolute; top: 8px; right: 12px; background: none; border: none; ' +
        'color: #ff6b6b; font-size: 24px; cursor: pointer; padding: 0; width: 24px; ' +
        'height: 24px; line-height: 24px; text-align: center; border-radius: 50%; ' +
        'transition: all 0.2s ease;';
      
      safeAddEventListener(closeButton, 'mouseenter', function() {
        closeButton.style.background = 'rgba(255, 107, 107, 0.2)';
      });
      
      safeAddEventListener(closeButton, 'mouseleave', function() {
        closeButton.style.background = 'none';
      });
      
      safeAddEventListener(closeButton, 'click', function() {
        panel.style.display = 'none';
      });
      
      panel.appendChild(closeButton);
      document.body.appendChild(panel);
      
      return panel;
    } catch (error) {
      console.error('Error creating UI:', error);
      return null;
    }
  }

  // Enhanced toggle button with better positioning
  function createToggleButton() {
    try {
      var button = document.createElement('button');
      button.id = 'fancy-text-toggle';
      button.innerHTML = '✨';
      button.title = 'Toggle Fancy Text Panel';
      button.style.cssText = 
        'position: fixed; top: 50%; right: 20px; transform: translateY(-50%); ' +
        'width: 56px; height: 56px; border-radius: 50%; ' +
        'background: linear-gradient(135deg, #ff6b6b, #ee5a52); border: none; ' +
        'color: white; font-size: 24px; cursor: pointer; z-index: 999998; ' +
        'box-shadow: 0 4px 20px rgba(255, 107, 107, 0.4); ' +
        'transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1); ' +
        'backdrop-filter: blur(10px);';

      safeAddEventListener(button, 'mouseenter', function() {
        button.style.transform = 'translateY(-50%) scale(1.1)';
        button.style.boxShadow = '0 6px 25px rgba(255, 107, 107, 0.6)';
      });

      safeAddEventListener(button, 'mouseleave', function() {
        button.style.transform = 'translateY(-50%) scale(1)';
        button.style.boxShadow = '0 4px 20px rgba(255, 107, 107, 0.4)';
      });

      safeAddEventListener(button, 'click', function() {
        button.style.transform = 'translateY(-50%) scale(0.95)';
        setTimeout(function() {
          button.style.transform = 'translateY(-50%) scale(1)';
        }, 150);
      });

      document.body.appendChild(button);
      return button;
    } catch (error) {
      console.error('Error creating toggle button:', error);
      return null;
    }
  }

  // Enhanced initialization with better error handling
  function initFancyTextExtension() {
    try {
      // Prevent multiple initializations
      if (isInitialized) return;
      
      // Only run on YouTube pages
      if (!window.location.hostname.includes('youtube.com')) {
        return;
      }

      // Wait for page to load
      if (document.readyState === 'loading') {
        safeAddEventListener(document, 'DOMContentLoaded', initFancyTextExtension);
        return;
      }

      // Cleanup existing elements
      var existingPanel = document.getElementById('fancy-text-panel');
      var existingButton = document.getElementById('fancy-text-toggle');
      if (existingPanel) existingPanel.remove();
      if (existingButton) existingButton.remove();

      // Create UI elements
      toggleButton = createToggleButton();
      extensionPanel = createFancyTextUI();

      if (!toggleButton || !extensionPanel) {
        console.error('Failed to create UI elements');
        return;
      }

      // Enhanced toggle functionality
      safeAddEventListener(toggleButton, 'click', function() {
        try {
          if (extensionPanel.style.display === 'none' || !extensionPanel.style.display) {
            extensionPanel.style.display = 'block';
            var input = document.getElementById('fancy-text-input');
            if (input) {
              setTimeout(function() { input.focus(); }, 100);
            }
          } else {
            extensionPanel.style.display = 'none';
          }
        } catch (error) {
          console.error('Error toggling panel:', error);
        }
      });

      // Enhanced Enter key handling
      var input = document.getElementById('fancy-text-input');
      if (input) {
        safeAddEventListener(input, 'keypress', function(e) {
          if (e.key === 'Enter' || e.keyCode === 13) {
            try {
              var inputText = validateInput(e.target.value);
              if (inputText) {
                var transformedText = fancyTextStyles.bold.transform(inputText);
                insertTextIntoChat(transformedText);
                e.target.value = '';
              } else {
                showNotification('Please enter some text first!');
              }
            } catch (error) {
              console.error('Error on Enter key:', error);
              showNotification('Error processing text. Please try again.');
            }
          }
        });
      }

      isInitialized = true;
      console.log('YouTube Fancy Text extension initialized successfully!');
      
    } catch (error) {
      console.error('Error initializing extension:', error);
    }
  }

  // Enhanced cleanup function
  function cleanup() {
    try {
      cleanupEventListeners();
      if (extensionPanel && extensionPanel.parentNode) {
        extensionPanel.parentNode.removeChild(extensionPanel);
      }
      if (toggleButton && toggleButton.parentNode) {
        toggleButton.parentNode.removeChild(toggleButton);
      }
      isInitialized = false;
    } catch (error) {
      console.error('Error during cleanup:', error);
    }
  }

  // Enhanced navigation detection for YouTube SPA
  function setupNavigationListener() {
    var lastUrl = location.href;
    var observer = new MutationObserver(function() {
      try {
        var url = location.href;
        if (url !== lastUrl) {
          lastUrl = url;
          cleanup();
          setTimeout(initFancyTextExtension, 1000);
        }
      } catch (error) {
        console.error('Error in navigation observer:', error);
      }
    });

    try {
      observer.observe(document, { subtree: true, childList: true });
    } catch (error) {
      console.error('Error setting up navigation observer:', error);
    }
  }

  // Add CSS animations
  function addAnimationCSS() {
    try {
      var style = document.createElement('style');
      style.textContent = 
        '@keyframes fadeIn { from { opacity: 0; transform: translateY(-10px); } to { opacity: 1; transform: translateY(0); } }' +
        '@keyframes fadeOut { from { opacity: 1; transform: translateY(0); } to { opacity: 0; transform: translateY(-10px); } }';
      document.head.appendChild(style);
    } catch (error) {
      console.error('Error adding CSS:', error);
    }
  }

  // Initialize everything
  function start() {
    try {
      addAnimationCSS();
      initFancyTextExtension();
      setupNavigationListener();
    } catch (error) {
      console.error('Error starting extension:', error);
    }
  }

  // Start the extension
  if (document.readyState === 'loading') {
    safeAddEventListener(document, 'DOMContentLoaded', start);
  } else {
    start();
  }

  // Cleanup on page unload
  safeAddEventListener(window, 'beforeunload', cleanup);

})();